/** @file   minigunai.cpp
 * @brief   Implementation of MinigunAi - class.
 * @version $Revision: 1.4 $
 * @date    $Date: 2006/04/03 20:57:50 $
 * @author  Tomi Lamminsaari
 */


#include "minigunai.h" // class's header file
#include "warglobals.h"
#include "utils.h"
#include "www_assert.h"
using namespace eng2d;

namespace WeWantWar {



/** Constructor.
 */
MinigunAI::MinigunAI( GameObject* pAlien ) :
  AIController( pAlien, WarGlobals::pObjTable->pPlayer ),
  
  m_alerted( false )
{
	WWW_ASSERT( pAlien != 0 );
	AIController::m_visualRange = 500;
	AIController::m_attackRange = 250;
	this->setFlags( AIController::WAIT_TARGET );
}

/** Destructor.
 */
MinigunAI::~MinigunAI()
{
	
}



/** The update-method
 */
void MinigunAI::updateAI( bool seeTarget, float dist )
{
  bool withinRange = this->withinAttackRange();
  bool weaponReady = !m_pObject->reloading();
  bool facingTarget = Utils::isFacingObject( m_pObject, m_pTarget, 20 );
  Vec2D ourPos = m_pObject->position();
  Vec2D targetPos = m_pTarget->position();
  int ourAngle = m_pObject->angle();
  
  if ( m_alerted == false ) {
    if ( seeTarget == true ) {
      m_alerted = true;
    } else {
      // We haven't seen player yet, so we stay put
      this->reset();
      this->setCounter( 5 + rand() % 5 );
      return;
    }
  }
  this->setRandomMovement( 10 + rand() % 10 );
  
  if ( seeTarget == true ) {
    if ( withinRange == true ) {
      if ( facingTarget == true ) {
        if ( weaponReady == true ) {
          // We're ready to shoot.
          this->reset();
          this->shoot() = ( rand()%10 < 8 ) ? 1 : 0;
          if ( this->shoot() == 0 ) {
            this->setRandomMovement( 15 + rand()%10 );
            return;
          }
          this->setCounter( 10 );
          this->turn() = (rand()%3) - 1;
          return;
        } else {
          // We're pointing to the target but our gun is not ready.
          this->setRandomMovement( 6 + (rand()%6) );
          return;
        }
        
      } else {
        // We're not facing the target. Keep turning towards him.
        this->reset();
        this->forward() = 1;
        int d = Utils::findTurningDir( ourPos, targetPos, ourAngle );
        this->turn() = 2*d;
        if ( dist < 80 ) {
          this->forward() = 0;
          this->turn() = 4*d;  // turn more quickly
        }
        this->setCounter(4 + rand()%8);
        return;
      }
      
    } else {
      // We're not within the range. Get closer. Use A* algorithm to find path
      // to the player.
      AStarAlgorithm algo( *this );
      algo.SetMaxRouteNodes( 64 );
      bool pathFound = algo.FindPath( ourPos, targetPos );
      const MPathFinderAlgo::NodeArray& nodes = algo.GetPathNodes();
      if ( pathFound == true && nodes.size() > 2 ) {
        // We found the route.
        Vec2D nodePos( nodes.at(1) );
        int d = Utils::findTurningDir( ourPos, nodePos, ourAngle );
        this->reset();
        this->forward() = 1;
        this->turn() = 2*d;
        this->setCounter( 4 + rand() % 3 );
        return;
      } else {
        // No path from our current position to the player. Move randomly.
        return;
      }
    }
  } else {
    // We don't see the target.
    AStarAlgorithm algo( *this );
    algo.SetMaxRouteNodes( 128 );
    bool pathFound = algo.FindPath( ourPos, targetPos );
    const MPathFinderAlgo::NodeArray& nodes = algo.GetPathNodes();
    if ( pathFound == true && nodes.size() > 2 ) {
      Vec2D nodePos( nodes.at(1) );
      int d = Utils::findTurningDir( ourPos, nodePos, ourAngle );
      this->reset();
      this->forward() = 1;
      this->turn() = 2*d;
      this->setCounter( 4 + rand() % 3 );
      return;
    } else {
      // No path to the target. Move randomly.
      return;
    }
  }
}

} // end of namespace

/**
 * Version history
 * ===============
 * $Log: minigunai.cpp,v $
 * Revision 1.4  2006/04/03 20:57:50  lamminsa
 * Lesser route nodes.
 *
 * Revision 1.3  2006/03/16 21:34:54  lamminsa
 * Behaviour tweaks.
 *
 * Revision 1.2  2006/02/26 07:36:43  lamminsa
 * New logic.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:43  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:15:35+02  lamminsa
 * Initial revision
 *
 */
 
